;------------------------------------------------------------------------
; Lets get going
; Cold and Warm program entry points
;------------------------------------------------------------------------

COLD            JSR     INIT            Initialise assembler
WARM            CLD                     You'll probably know why
                JSR     CROUT           Print CR before we do anything

                .DO     PROCESSOR=$6502
                LDY     #0              Clear auto flag
                STY     AUTO_FLAG
                .EL
                STZ     AUTO_FLAG       Clear auto flag
                .FI

;------------------------------------------------------------------------
; Get input line (main program loop)
;------------------------------------------------------------------------

GETLINE         LDX     #$FF            Initialise stack each time
                TXS

                INX                     Clear index in input buffer
                LDA     #PROMPT         Print the assembler prompt
                JSR     ECHO
                JSR     AUTONUM         Generate line number if auto=on

.IN             LDA     KBDCR           Was a key pressed?
                BPL     .IN             Nope!
                LDA     KBD             Get the key

                CMP     #ESC            Is it the ESC key?
                BNE     .NOTESC         Nope!

                .DO     PROCESSOR=$6502
                LDA     #0              Clear auto flag
                STA     AUTO_FLAG
                .EL
                STZ     AUTO_FLAG       Clear auto flag
                .FI

.CANCEL         LDY     #STR_CANCEL     Print \ and a CR
                JSR     PRINT
                BPL     GETLINE         Always taken! Restart line

.NOTESC         CMP     #BS             Back space key then?
                BNE     .NOTBS          Nope!
                DEX                     Decrement input pointer
                BPL     .IN             Still positive!
                INX                     Don't allow it to go negative
                BPL     .IN             Always taken!

.NOTBS          CMP     #TAB            Is it the TAB character?
                BNE     .NOTTAB         Nope!
                LDY     #0              Point to 1st tab stop
.TABS           LDA     TABS,Y          It X smaller than this one?
                STA     COUNT
                LDA     #" "
                CPX     COUNT
                BCC     .DOTAB          Yes! Tab to this position
                INY
                CPY     #NUMTABS        Maximum number of tab stops
                BCC     .TABS           Try next!

.DOTAB          STA     IN,X            Save this character
                INX
                JSR     ECHO             and echo it
                CPX     COUNT           At tab stop?
                BCC     .DOTAB          Not yet!
                BCS     .IN             Always taken!

.NOTTAB         CMP     #CR             Is it a CR then?
                BEQ     .CR             Yes!
                CMP     #" "            Don't allow any other CTRL chars
                BCC     .IN             Ignore this CTRL char!
                STA     IN,X            Save new character in input buffer
                JSR     ECHO             and print it
                INX
                BPL     .IN             Get next character!
                BMI     .CANCEL         Always taken! Too many chars.

.CR             STA     IN,X            Mark end of line now
                JSR     ECHO
                JSR     FIRSTNONSPC     Find first non space
                BEQ     GETLINE         End of line reached!

                CMP     #"0"            Is it a line number?
                BCC     .NOLINE         No!
                CMP     #"9"+1
                BCC     .LINENUM        It is! Add it to the program

.NOLINE         JSR     COMMAND         Parse command, or line number

.CHK_ERROR      LDA     ERROR           Was there an error?
                BEQ     .GETLINE        Nope!
                JSR     PRINT_ERROR     Print error
.GETLINE        JMP     GETLINE         Stay in main program loop

.LINENUM        JSR     ADDLINE         Add line to program
                JMP     .CHK_ERROR      Check if an error occurred

TABS            .DA     #TAB1           1st tab stop
                .DA     #TAB2           2nd tab stop
                .DA     #TAB3           3rd tab stop
NUMTABS         .EQ     3               Number of tab stops

;------------------------------------------------------------------------
; Move a part of memory down (to delete or shorten source lines)
;
; SRCE  2       Address of source data (destroyed)
; DEST  2       Address of destination data (destroyed)
; LENG  2       Holds the number of bytes to move
;------------------------------------------------------------------------


MOV_DOWN        LDY     #0              Clear index in 256 byte blocks
                LDX     LENG+1          Get number of blocks to be moved
                BEQ     .LAST           Move last (partial) block

.LOOP1          LDA     (SRCE),Y        Move this byte
                STA     (DEST),Y
                INY                     Point to next byte in block
                BNE     .LOOP1          Block not done yet!
                INC     SRCE+1          Point to next block
                INC     DEST+1
                DEX                     Count down blocks
                BNE     .LOOP1          Not all blocs done yet!

.LAST           LDX     LENG            Count remainder of last block
                BEQ     .EXIT           Oh, there is no remainder!

.LOOP2          LDA     (SRCE),Y        Move this byte
                STA     (DEST),Y
                INY                     Point to next byte in last block
                DEX                     Count down the bytes
                BNE     .LOOP2          Last block not done yet!
.EXIT           RTS

;------------------------------------------------------------------------
; Move a part of memory up (to make room for new or longer source line)
;
; SRCE  2       Address of source data
; DEST  2       Address of destination data
; LENG  2       Holds number of bytes to move
;
;------------------------------------------------------------------------

MOV_UP          LDX     LENG+1          We have started the move at the
                CLC                      end otherwise we risk
                TXA                       overwriting the source
                ADC     SRCE+1
                STA     SRCE+1          So add Length-High to source and
                CLC                      destination address
                TXA
                ADC     DEST+1
                STA     DEST+1

                INX                     Allow BNE to signal the end
                LDY     LENG
                BEQ     .PAGE           Only entire pages to be moved!
                DEY
                BEQ     .PART           Move last, partial, page first!

.LOOP1          LDA     (SRCE),Y        Move one entire page
                STA     (DEST),Y
                DEY                      backwards
                BNE     .LOOP1          Page not done yet
.PART           LDA     (SRCE),Y        Move first byte of page too
                STA     (DEST),Y
.PAGE           DEY
                DEC     SRCE+1          Decrement source and destination
                DEC     DEST+1
                DEX
                BNE     .LOOP1          Move all pages!
                RTS

;------------------------------------------------------------------------
; Print error
;------------------------------------------------------------------------

PRINT_ERROR     LDY     #5              Print ***
                JSR     .PRINT
                LDY     ERROR           Followed by error string
                JSR     .PRINT          Print 3 characters
                LDY     #0              Print ERR behind it
                STY     ERROR           Clear the error now

.PRINT          LDA     .ERRORS,Y       Get character
                JSR     ECHO             and print it
                INY                     Increment pointer
                ASL                     End of string now?
                BCS     .PRINT          Not done yet!
                RTS

.ERRORS         .AT     -/ ERR/,#CR      0
                .AT     -/*** /          5
ERR_SYN = 9
                .AT     -/SYN/          Syntax error
ERR_LBL = 12
                .AT     -/LBL/          Label error / no global
ERR_RNG = 15
                .AT     -/RNG/          Range error
ERR_MIS = 18
                .AT     -/MIS/          Missing parameter/operand
ERR_DIV = 21
                .AT     -/DIV/          Divide by 0 error
ERR_MEM = 24
                .AT     -/MEM/          Memory full / illegal TA
ERR_DEF = 27
                .AT     -/DEF/          Undefined label
ERR_DIR = 30
                .AT     -/DIR/          Illegal directive error
ERR_OPE = 33
                .AT     -/OPE/          Operand error
ERR_MNE = 36
                .AT     -/MNE/          Mnemonic error
ERR_EXT = 39
                .AT     -/EXT/          Extra definition error
                .DO     PROCESSOR=$6502
ERR_OLD = 42
                .AT     -/OLD/          Old error (OLD not possible)
                .FI

;------------------------------------------------------------------------
; Find next non space
; Z=1 if EOL found
;------------------------------------------------------------------------

FIRSTNONSPC     LDX     #0              Start at the begin of the line
NNONSPC         LDA     IN,X            Is this a space?
                CMP     #" "
                BNE     .NOSPACE        It is not!
                INX
                BNE     NNONSPC         Always taken!

.NOSPACE        CMP     #CR             Set Z if we're at EOL
                RTS

;------------------------------------------------------------------------
; Print a dot
;------------------------------------------------------------------------

PRDOT           LDA     #"."            Print a dot
                BNE     ECHO2

;------------------------------------------------------------------------
; Print a space
;------------------------------------------------------------------------

PRSPACE         LDA     #" "            Print a space
                BNE     ECHO2

;------------------------------------------------------------------------
; Print a CR
;------------------------------------------------------------------------

CROUT           LDA     #CR             Print a CR
ECHO2           JMP     ECHO

;------------------------------------------------------------------------
; Print word in hex
; A holds MSB, Y holds LSB, C=1 print dollar
;------------------------------------------------------------------------

PRWORD          PHA                     Save MSB
                LDA     #"$"            Print dollar symbol
                JSR     ECHO
                PLA
                JSR     PRBYTE          Print MSB
                TYA
                JMP     PRBYTE          Print LSB

;------------------------------------------------------------------------
; Initialise the assembler before we can start
;------------------------------------------------------------------------

INIT            CLD                     You'll never know
                LDY     #STR_WELCOME    Print welcome string
                JSR     PRINT

                LDX     #PASS           Clear zero page
                LDY     #0
.CLEAR          STY     0,X             Clear byte
                INX
                BNE     .CLEAR          Not all done yet!

                LDA     #DEF_LOMEM      Set default lomem and hihem
                STA     LOMEM+1          (Low bytes are $00)
                STA     HIMEM+1         Himem will grow!

.LOOP           LDA     HIMEM+1         Stop if we run into our own code
                CMP     /COLD
                BEQ     .THATSIT

                LDA     (HIMEM),Y       See if this memory page exists
                TAX                     Save original value

                LDA     #%0101.0101     Try this value first
                JSR     .PROBE
                BNE     .THATSIT        End of RAM!

                LDA     #%1010.1010     Then try this value
                JSR     .PROBE
                BNE     .THATSIT        End of RAM!

                TXA                     Restore original value
                STA     (HIMEM),Y

                INC     HIMEM+1         Try next memory page
                BPL     .LOOP           No need to proceed beyond $8000!

.THATSIT        JSR     DONEW           Do new command

                LDA     #.XEC_RTS       There is no last eXec address yet
                STA     XEC_LAST
                LDA     /.XEC_RTS
                STA     XEC_LAST+1
                RTS

.PROBE          STA     (HIMEM),Y       Store this value
                NOP                     Allow it to settle
                CMP     (HIMEM),Y       See if it was accepted
.XEC_RTS        RTS

;------------------------------------------------------------------------
; Get a single 16-bit value from input line
;------------------------------------------------------------------------

                .DO     PROCESSOR=$6502
GET_VAL         LDA     #0              Clear negative flag
                STA     NEG_FLAG
                STA     HEXVAL          Clear end result
                STA     HEXVAL+1
                STA     DELIM           Clear digit entered flag
                .EL
GET_VAL         STZ     NEG_FLAG        Clear negative flag
                STZ     HEXVAL          Clear end result
                STZ     HEXVAL+1
                STZ     DELIM           Clear digit entered flag
                .FI

                LDA     IN,X            Read character from IN
                CMP     #"+"            Is it explicit positive?
                BEQ     .POS            Yes!
                CMP     #"-"            Is it a negative number?
                BNE     .NOSIGN         No! No prefix sign given
                STA     NEG_FLAG        Set negative flag
.POS            INX                     Point to next byte in IN
                LDA     IN,X             and get it

.NOSIGN         CMP     #"0"            Could it be a decimal number?
                BCC     .HEX            Nope!
                CMP     #"9"+1
                BCS     .HEX            Nope!

;------------------------------------------------------------------------
; Convert a decimal number

.DECLOOP        EOR     #"0"            Strip ASCII part
                CMP     #9+1            Is it still a decimal digit?
                BCS     .DONEG1         Nope!
                TAY                     Save new digit
                ASL     HEXVAL          Multiply previous value by 10
                ROL     HEXVAL+1
                BCS     .RANGE          Exit with range error!
                LDA     HEXVAL
                ASL
                STA     DEC_SAVE
                LDA     HEXVAL+1
                ROL
                BCS     .RANGE          Exit with range error!
                ASL     DEC_SAVE
                ROL
                STA     DEC_SAVE+1
                BCS     .RANGE          Exit with range error!
                LDA     DEC_SAVE
                ADC     HEXVAL
                STA     HEXVAL
                LDA     DEC_SAVE+1
                ADC     HEXVAL+1
                STA     HEXVAL+1
                BCS     .RANGE          Exit with range error!
                TYA                     Add new digit to result
                ADC     HEXVAL
                STA     HEXVAL
                BCC     .NOCY           No carry to high byte
                INC     HEXVAL+1
                BEQ     .RANGE          Exit with range error!
.NOCY           INX                     Point to next digit
                LDA     IN,X             and get it
                BNE     .DECLOOP        Always taken!

;------------------------------------------------------------------------
; Convert a hex number

.HEX            CMP     #"$"            Is it a hex number?
                BNE     .BIN            Nope!

.HEXLOOP        INX                     Get next character from IN
                LDA     IN,X
                EOR     #"0"            Ignore ASCII part for decimals
                CMP     #9+1
                BCC     .DIG            It was a decimal digit!
                ADC     #$88            Map letters A - F to $FA-FF
                CMP     #$FA
                BCC     .NOTHEX         No hex digit anymore!
.DIG            ASL                     Left justify new nibble
                ASL
                ASL
                ASL
                LDY     #4              Set shift counter
                STY     DELIM           Indicate that input is correct
.HEXSHIFT       ASL                     Shift new bit into result
                ROL     HEXVAL
                ROL     HEXVAL+1
                BCS     .RANGE          Exit with range error!
                DEY                     Decrement counter
                BNE     .HEXSHIFT       Do all 4 bits
                BEQ     .HEXLOOP        Do next nibble (if any)!

.NOTHEX         LDA     DELIM           Was a digit entered?
                BNE     .DONEG1         Yes!
                LDA     PC_BEG          If not it was the PC value!
                STA     HEXVAL
                LDA     PC_BEG+1
                STA     HEXVAL+1
.DONEG1         JMP     .DONEG          Check negative flag & we're done

.RANGE          JMP     RANGE_ERROR     Exit with range error

;------------------------------------------------------------------------
; Convert a binary number

.BIN            CMP     #"%"            Is it a binary number?
                BNE     .PASC           Nope!

.BINLOOP        INX                     Point to next char
                LDA     IN,X             and get it
                CMP     #"."            Ignore dots (for human eyes only)
                BEQ     .BINLOOP
                EOR     #"0"            Ignore ASCII part
                LSR                     Shift bit to Carry
                BNE     .EXIT           End of binary number
                ROL     HEXVAL          Roll new bit into end result
                ROL     HEXVAL+1
                BCS     .RANGE          Range error!
                INC     DELIM           Signal that bits were entered
                BNE     .BINLOOP        Always taken! (reasonably)

.EXIT           LDA     DELIM           Was a number entered?
                BNE     .DONEG          Yes! Check negative flag now
                JMP     DIR_OPE         Exit with operand error

;------------------------------------------------------------------------
; Now it must be a label!

.LABEL          JSR     FIND_LABEL      Find this label
                BCS     .OPE_ERROR      Exit with operand error!
                BEQ     .NOT_FOUND      Label not found!
                LDA     LABEL_FLAG      Copy assigned flag to
                ORA     FORWARD          forward referenced flag
                STA     FORWARD
.LBL_EXIT       JMP     .DONEG          Check negative flag & we're done

.NOT_FOUND      STA     HEXVAL          A=0, clear hex val
                STA     HEXVAL+1
                DEC     UNDEF           Set undefined label flag
                BNE     .LBL_EXIT       Always taken!

;------------------------------------------------------------------------
; It's none of the above! Operand error

.OPE_ERROR      JMP     DIR_OPE         Exit with operand error

;------------------------------------------------------------------------
; Convert a positive ASCII

.PASC           CMP     #"'"            Is it a positive ASCII?
                BNE     .NASC           Nope!

                JSR     .GETASC         Get ASCII character
                AND     #%0111.1111     Make it positive
                BPL     .ASC_END        Always taken!

;------------------------------------------------------------------------
; Convert a negative ASCII

.NASC           CMP     #"""            Is it negative ASCII?
                BNE     .LABEL          Nope!
                JSR     .GETASC         Get ASCII character
.ASC_END        STA     HEXVAL          And save it
;                                       Falls through to .DONEG

;------------------------------------------------------------------------
; Check negative flag and we're done

.DONEG          LDA     NEG_FLAG        Is the negative flag set?
                BEQ     .RTS            Nope!
                JSR     NEGATE          Make hexval negative

.RTS            LDA     ERROR           Set error flag upon exit
                PHP                     Save status
                LDA     HEXVAL+1        Preload new value
                LDY     HEXVAL
                PLP                     Restore status
                RTS

;------------------------------------------------------------------------

.GETASC         STA     DELIM           Save delimiter char ( ' or " )
                INX                     Point to actual ASCII character
                LDA     IN,X             and get it
                CMP     #CR             Replace CR by a space
                BNE     .NOTCR          Don't bother!
                LDA     #" "
.NOTCR          TAY                     Temporarily save character
                INX                     See if ASCII is terminated by
                LDA     IN,X             delimiter
                CMP     DELIM
                BNE     .DELIM          No, don't bother to increment
                INX                      pointer once more
.DELIM          TYA                     Get character back
                RTS

;------------------------------------------------------------------------
; Get expression
;------------------------------------------------------------------------

GET_EXPRES      JSR     GET_VAL         Get value
                BNE     .RTS            An error occurred!

.NEXT           STA     EXP_SAVE+1      Save this value for later
                STY     EXP_SAVE
                LDA     IN,X            Did an operator follow the value?
                INX
                CMP     #"+"
                BEQ     .ADD            Addition!
                CMP     #"-"
                BEQ     .SUB            Subtraction!
                CMP     #"*"
                BEQ     .MUL            Multiplication!
                CMP     #"/"
                BEQ     .DIV            Division!
                DEX                     End of expression. One too far
                LDA     HEXVAL+1        Load value now
                LDY     HEXVAL
                CPY     HEXVAL          Make Z=0 (No errors)
.RTS            RTS                     None of the above! We're done!

.ADD            JSR     GET_VAL         Get next value
                BNE     .RTS            An error occurred!
                CLC                     Perform add function
                LDA     HEXVAL
                ADC     EXP_SAVE
                STA     HEXVAL
                TAY
                LDA     HEXVAL+1
                ADC     EXP_SAVE+1
                STA     HEXVAL+1
                JMP     .NEXT           More operands can follow

.SUB            JSR     GET_VAL         Get next value
                BNE     .RTS            An error occurred!
                SEC                     Perform subtract function
                LDA     EXP_SAVE
                SBC     HEXVAL
                STA     HEXVAL
                TAY
                LDA     EXP_SAVE+1
                SBC     HEXVAL+1
                STA     HEXVAL+1
                JMP     .NEXT           More operands can follow!

.MUL            JSR     GET_VAL         Get next value
                BNE     .RTS            An error occurred!

                .DO     PROCESSOR=$6502
                LDA     #0              Clear temp result
                STA     MULDIV
                STA     MULDIV+1
                .EL
                STZ     MULDIV          Clear temp result
                STZ     MULDIV+1
                .FI
                LDY     #16             Do 16-bit multiply

.MULLOOP        LDA     HEXVAL          Get LSB of operand 2 in carry
                LSR
                BCC     .MULZERO        Multiply by 0! Forget about it
                CLC                     Add shifted operand 1 to temp
                LDA     MULDIV
                ADC     EXP_SAVE
                STA     MULDIV
                LDA     MULDIV+1
                ADC     EXP_SAVE+1
                STA     MULDIV+1
.MULZERO        ROR     MULDIV+1        Roll temp result bit into
                ROR     MULDIV           end result
                ROR     HEXVAL+1
                ROR     HEXVAL
                DEY
                BNE     .MULLOOP        Do all 16 bits!

                LDY     HEXVAL          Preload the value
                LDA     HEXVAL+1
                JMP     .NEXT           More operands can follow!

.DIV            JSR     GET_VAL         Get next value
                BNE     .RTS            An error occurred
                LDA     HEXVAL          Avoid divide by 0
                ORA     HEXVAL+1
                BEQ     .DIVERR         It is divide by 0!

                .DO     PROCESSOR=$6502
                LDA     #0              Clear temp result
                STA     MULDIV
                STA     MULDIV+1
                .EL
                STZ     MULDIV          Clear temp result
                STZ     MULDIV+1
                .FI
                LDY     #16             Do 16-bit multiply

.DIVLOOP        ASL     EXP_SAVE        Roll 1 bit out of operand 1
                ROL     EXP_SAVE+1
                ROL     MULDIV
                ROL     MULDIV+1
                SEC                     Does operand 2 go into temp?
                LDA     MULDIV
                SBC     HEXVAL
                PHA                     Don't save temp yet
                LDA     MULDIV+1
                SBC     HEXVAL+1
                BCC     .DIVSKIP        It didn't go!
                INC     EXP_SAVE        Set low bit of result
                STA     MULDIV+1        Save new temp result
                PLA                     Save low byte too without
                PHA                      affecting the stack
                STA     MULDIV
.DIVSKIP        PLA                     Clear stack
                DEY
                BNE     .DIVLOOP        Do all 16 bits!

                LDY     EXP_SAVE        Preload the value
                STY     HEXVAL
                LDA     EXP_SAVE+1
                STA     HEXVAL+1
.NEXT_STEP      JMP     .NEXT           More operands can follow!

.DIVERR         LDA     PASS            Ignore div 0 during pass 1
                BEQ     .NEXT_STEP      It is pass 0!
                LDA     #ERR_DIV        Exit with divide by 0 error
                STA     ERROR
                RTS

;------------------------------------------------------------------------
; Print decimal number and/or write value to IN,X
; HEXVAL  holds the value to be printed
;------------------------------------------------------------------------

PRDECI          LDA     HEXVAL          Copy value to work registers
                STA     EXP_SAVE
                LDA     HEXVAL+1
                STA     EXP_SAVE+1
                LDY     #4              A maximum of 5 decades and 4
                STY     LEAD0            leading zeroes

.DECLOOP        LDA     #"0"            Start with 0
                STA     CHAR
.LOOP           LDA     EXP_SAVE        Try to subtract decade from
                CMP     .DECIL,Y         value
                LDA     EXP_SAVE+1
                SBC     .DECIH,Y
                BCC     .DONE           Didn't go! Done this decade
                STA     EXP_SAVE+1      Save new intermediate result
                LDA     EXP_SAVE
                SBC     .DECIL,Y
                STA     EXP_SAVE
                INC     CHAR            Increment counter
                BNE     .LOOP

.DONE           LDA     CHAR            Get decade character
                INY                     Last digit?
                DEY
                BEQ     .PRINT          Yes! Simply print it
                CMP     #"0"            Is it a 0?
                BEQ     .LEAD0          Yep!
                STA     LEAD0           B7=1, no more leading zeroes

.LEAD0          BIT     LEAD0           Still leading zeroes?
                BPL     .NEXT           Yes! Skip them

.PRINT          JSR     ECHO            Print character
                BIT     AUTO_FLAG       Auto line number?
                BPL     .NEXT           Nope!
                STA     IN,X            Save character in input line
                INX
.NEXT           DEY                     Do all 5 digits (except the LSD)
                BPL     .DECLOOP        Not done them all yet!
                RTS

.DECIL          .DA     #1,#10,#100,#1000,#10000
.DECIH          .DA     /1,/10,/100,/1000,/10000

;------------------------------------------------------------------------
; Halt if key is pressed
;------------------------------------------------------------------------

HALT            LDA     KBDCR           Was a key pressed?
                BPL     .NOKEY          Nope!
                JSR     .ESC            Escape pressed?
                BEQ     HALT_RTS        Yes! Quit

.LOOP           LDA     KBDCR           Wait until next key is pressed
                BPL     .LOOP           No key pressed yet!
.ESC            LDA     KBD             Get the key
.NOKEY          CMP     #ESC            Make Z=1 if ESC
HALT_RTS        RTS                     Cancel if Z=1

;------------------------------------------------------------------------
; HELP command
;------------------------------------------------------------------------

CMD_HLP         LDY     #STR_WWW        Print help string
;                                       Fall through to PRINT

;------------------------------------------------------------------------
; Print some strings
;------------------------------------------------------------------------

PRINT           LDA     TEXT,Y          Get character
                JSR     ECHO
                INY
                ASL
                BCS     PRINT           Not done yet!
                JMP     CROUT           Exit with CR

TEXT
                .DO     PROCESSOR=$6502
PNTR_WELCOME    .AT     -#CR,/A1-ASM V1.0/
                .EL
PNTR_WELCOME    .AT     -#CR,/A1-ASM C1.0/
                .FI
PNTR_ERRORS     .AT     -/ ERRORS/
PNTR_WWW        .AT     -/WWW.SBPROJECTS.COM/
PNTR_CANCEL     .AT     -/\/

STR_WELCOME     .EQ     PNTR_WELCOME-TEXT
STR_ERRORS      .EQ     PNTR_ERRORS-PNTR_WELCOME
STR_WWW         .EQ     PNTR_WWW-PNTR_WELCOME
STR_CANCEL      .EQ     PNTR_CANCEL-PNTR_WELCOME

;------------------------------------------------------------------------
; Print object range
;------------------------------------------------------------------------

PRINT_OBJ       LDA     TA              See if code was generated
                EOR     TA_BEGIN
                BNE     .OK             Yes!
                LDA     TA+1
                EOR     TA_BEGIN+1
                BEQ     HALT_RTS        No code!
.OK             LDY     TA_BEGIN        First print begin address
                LDA     TA_BEGIN+1
                JSR     PRWORD
                JSR     PRDOT
                SEC                     Then print end address
                LDA     TA+1             (but this time one less)
                LDY     TA
                BNE     .NOBOR          No borrow from high byte
                SBC     #1
.NOBOR          DEY                     Decrement low byte
                JSR     PRWORD
                JMP     CROUT           Followed by CR

;------------------------------------------------------------------------
; Make HEXVAL negative
;------------------------------------------------------------------------

                .DO     PROCESSOR=$6502
NEGATE          LDA     HEXVAL          Negate value
                EOR     #$FF
                STA     HEXVAL
                LDA     HEXVAL+1
                EOR     #$FF
                STA     HEXVAL+1
                INC     HEXVAL
                BNE     .RTS            No carry to high byte!
                INC     HEXVAL+1
.RTS            RTS
                .EL
NEGATE          LDA     HEXVAL          Negate value
                EOR     #$FF
                STA     HEXVAL
                LDA     HEXVAL+1
                EOR     #$FF
                INC     HEXVAL
                BNE     .RTS            No carry to high byte!
                INC
.RTS            STA     HEXVAL+1
                RTS
                .FI
